<?php

namespace App\Http\Controllers\CANTEEN;

use App\Http\Controllers\Controller;
use App\Models\Unit;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Validation\Rule;

class UnitController extends Controller implements HasMiddleware
{
    public static function middleware(): array
    {
        return [
            new Middleware('permission:units.read', only: ['index']),
            new Middleware('permission:units.create', only: ['create', 'store']),
            new Middleware('permission:units.update', only: ['edit','update']),
            new Middleware('permission:units.delete', only: ['destroy']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        //
        $rowsQuery = Unit::sortable()
            ->where('user_main_id', auth()->user()->main_id);
        
        if($request->query('query')){

            $searchQuery = $request->query('query');

            $rowsQuery->where(function($query) use($searchQuery){
                $query->where('uid', 'like', '%' . $searchQuery . '%')
                ->orWhere('name', 'like', '%' . $searchQuery . '%')
                ->orWhere('symbol', 'like', '%' . $searchQuery . '%')
                ->orWhere('description', 'like', '%' . $searchQuery . '%');
            });
        }

        $rows = $rowsQuery->paginate(setting("data_per_page"));

        return view('canteen.pages.units.index', compact('rows'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        return view('canteen.pages.units.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
        $request->validate([
            'name' => ['required', 'max:100', Rule::unique('units')],
            'symbol' => ['required', 'max:100', Rule::unique('units')],
            'description' => ['nullable', 'string', 'max:255'],
        ]);

        $unit = new Unit();
        $unit->user_main_id = auth()->user()->main_id;
        $unit->name = $request->name;
        $unit->symbol = $request->symbol;
        $unit->description = $request->description;
        $unit->save();
        
        \Session::flash("message", "Unit created successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/units'),
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $subdomain, string $id)
    {
        //
        $unit = Unit::where('user_main_id', auth()->user()->main_id)->findOrFail($id);

        return view('canteen.pages.units.edit', compact('unit'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $subdomain, string $id)
    {
        //
        $request->validate([
            'name' => ['required', 'max:100', Rule::unique('units')->ignore($id)],
            'symbol' => ['required', 'max:100', Rule::unique('units')->ignore($id)],
            'description' => ['nullable', 'string', 'max:255'],
        ]);

        $unit = Unit::where('user_main_id', auth()->user()->main_id)->find($id);
        $unit->user_main_id = auth()->user()->main_id;
        $unit->name = $request->name;
        $unit->symbol = $request->symbol;
        $unit->description = $request->description;
        $unit->save();
        
        \Session::flash("message", "Unit updated successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/units'),
        ]);
    }

    public function beforeDelete(string $subdomain, string $id){

        return response()->json([
            'deletion_allowed' => true,
            'message' => ''
        ]);

        // return response()->json([
        //     'deletion_allowed' => false,
        //     'message' => 'This tax is associated with an item. You cannot remove it.',
        // ]);

    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
        $unit = Unit::where('user_main_id', auth()->user()->main_id)->findOrFail($id);
        $unit->delete();

        \Session::flash("message", "Unit deleted successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/units'),
        ]);
    }
}
