<?php

namespace App\Http\Controllers\CANTEEN;

use App\Exports\CashTransactionsExport;
use App\Exports\OnlineTransactionsExport;
use App\Exports\QRCodeTransactionsExport;
use App\Http\Controllers\Controller;
use App\Models\OrderPayment;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Maatwebsite\Excel\Facades\Excel;

class PaymentTransactionController extends Controller implements HasMiddleware
{
    public static function middleware(): array
    {
        return [
            new Middleware('permission:reports.cash-transections-read', only: ['cashTransactions']),
            new Middleware('permission:reports.qrcode-transections-read', only: ['qrcodeTransactions']),
            new Middleware('permission:reports.online-transections-read', only: ['onlineTransactions']),
        ];
    }

    public function cashTransactions(Request $request){

        //Filters Data
        $filter_users = User::select('id', 'uid', 'name', 'main_id')
            ->whereHas('accepted_order_payments')
            ->where('main_id', auth()->user()->main_id)
            ->get();

        //Data Query
        $rowsQuery = OrderPayment::sortable()
                ->with(['accepted_by:id,name', 'order:id,uid'])
                ->where('user_main_id', auth()->user()->main_id)
                ->where('payment_method_id', 1);

        if($request->query('query')){

            $searchQuery = $request->query('query');

            $rowsQuery->where(function($query) use($searchQuery){
                $query->where('uid', 'like', '%' . $searchQuery . '%')
                ->orWhere('amount', 'like', '%' . $searchQuery . '%')
                ->orWhere('status', 'like', '%' . $searchQuery . '%')
                ->orWhereHas('accepted_by', function($query) use($searchQuery){
                    $query->where('name', 'like', '%' . $searchQuery . '%');
                })
                ->orWhereHas('order', function($query) use($searchQuery){
                    $query->where('uid', 'like', '%' . $searchQuery . '%');
                });
            });
        }

        if($request->query('payment_accepted_by')){
            $rowsQuery->where('payment_accepted_by', $request->payment_accepted_by);
        }

        if ($request->query('from')) {
            $rowsQuery->whereDate('created_at', '>=', $request->from);
        }

        if ($request->query('to')) {
            $rowsQuery->whereDate('created_at', '<=', $request->to);
        }

        if($request->query('export')){
            $data = $rowsQuery->get()->map(function($row){
                return [
                    $row->uid,
                    $row->order->uid,
                    $row->accepted_by->name,
                    $row->currency,
                    $row->amount,
                    $row->status,
                    $row->created_at
                ];
            });

            return Excel::download(new CashTransactionsExport($data), 'Cash Transactions '.date("Y-m-d H:i:s").'.xlsx');
        }

        $rows = $rowsQuery->paginate(setting("data_per_page"));

        return view('canteen.pages.paymentTransaction.cash', compact('rows', 'filter_users'));
    }

    public function qrcodeTransactions(Request $request){

        //Filters Query
        $filter_users = User::select('id', 'uid', 'name', 'main_id')
            ->whereHas('accepted_order_payments')
            ->where('main_id', auth()->user()->main_id)
            ->get();

        //Data Query
        $rowsQuery = OrderPayment::sortable()
                ->with(['accepted_by:id,name', 'order:id,uid'])
                ->where('user_main_id', auth()->user()->main_id)
                ->where('payment_method_id', 3);

        if($request->query('query')){

            $searchQuery = $request->query('query');

            $rowsQuery->where(function($query) use($searchQuery){
                $query->where('uid', 'like', '%' . $searchQuery . '%')
                ->orWhere('amount', 'like', '%' . $searchQuery . '%')
                ->orWhere('status', 'like', '%' . $searchQuery . '%')
                ->orWhereHas('accepted_by', function($query) use($searchQuery){
                    $query->where('name', 'like', '%' . $searchQuery . '%');
                })
                ->orWhereHas('order', function($query) use($searchQuery){
                    $query->where('uid', 'like', '%' . $searchQuery . '%');
                });
            });
        }

        if($request->query('payment_accepted_by')){
            $rowsQuery->where('payment_accepted_by', $request->payment_accepted_by);
        }

        if ($request->query('from')) {
            $rowsQuery->whereDate('created_at', '>=', $request->from);
        }

        if ($request->query('to')) {
            $rowsQuery->whereDate('created_at', '<=', $request->to);
        }

        if($request->query('export')){
            $data = $rowsQuery->get()->map(function($row){
                return [
                    $row->uid,
                    $row->order->uid,
                    $row->accepted_by->name,
                    $row->currency,
                    $row->amount,
                    $row->status,
                    $row->created_at
                ];
            });

            return Excel::download(new QRCodeTransactionsExport($data), 'QR Code Transactions '.date("Y-m-d H:i:s").'.xlsx');
        }

        $rows = $rowsQuery->paginate(setting("data_per_page"));

        return view('canteen.pages.paymentTransaction.qrcode', compact('rows', 'filter_users'));

    }

    public function onlineTransactions(Request $request){

        //Filters Query
        $filter_users = User::select('id', 'uid', 'name', 'main_id')
            ->whereHas('accepted_order_payments')
            ->where('main_id', auth()->user()->main_id)
            ->get();

        //Data Query
        $rowsQuery = OrderPayment::sortable()
                ->with(['accepted_by:id,name', 'order:id,uid'])
                ->where('user_main_id', auth()->user()->main_id)
                ->where('payment_method_id', 2);

        if($request->query('query')){

            $searchQuery = $request->query('query');

            $rowsQuery->where(function($query) use($searchQuery){
                $query->where('uid', 'like', '%' . $searchQuery . '%')
                ->orWhere('amount', 'like', '%' . $searchQuery . '%')
                ->orWhere('status', 'like', '%' . $searchQuery . '%')
                ->orWhereHas('accepted_by', function($query) use($searchQuery){
                    $query->where('name', 'like', '%' . $searchQuery . '%');
                })
                ->orWhereHas('order', function($query) use($searchQuery){
                    $query->where('uid', 'like', '%' . $searchQuery . '%');
                });
            });
        }

        if($request->query('payment_accepted_by')){
            $rowsQuery->where('payment_accepted_by', $request->payment_accepted_by);
        }

        if ($request->query('from')) {
            $rowsQuery->whereDate('created_at', '>=', $request->from);
        }

        if ($request->query('to')) {
            $rowsQuery->whereDate('created_at', '<=', $request->to);
        }

        if($request->query('export')){
            $data = $rowsQuery->get()->map(function($row){
                return [
                    $row->uid,
                    $row->order->uid,
                    $row->accepted_by->name,
                    $row->currency,
                    $row->amount,
                    $row->status,
                    $row->created_at
                ];
            });

            return Excel::download(new OnlineTransactionsExport($data), 'Online Transactions '.date("Y-m-d H:i:s").'.xlsx');
        }

        $rows = $rowsQuery->paginate(setting("data_per_page"));

        return view('canteen.pages.paymentTransaction.online', compact('rows', 'filter_users'));

    }

}
