<?php

namespace App\Http\Controllers\CANTEEN;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Customer;
use App\Models\Item;
use App\Models\ItemAddon;
use App\Models\Order;
use App\Models\OrderStatus;
use App\Models\PaymentMethod;
use App\Models\PaymentStatus;
use App\Models\Tax;
use App\Models\User;
use App\Traits\OrderUtils;
use Carbon\Carbon;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Http\Request;
use stdClass;

class POSController extends Controller implements HasMiddleware
{
    public static function middleware(): array
    {
        return [
            new Middleware('permission:orders.create', only: ['index','orderStore']),
        ];
    }

    use OrderUtils;

    public function index(){

        $canteen = User::find(auth()->user()->main_id);

        $categories = Category::whereHas('items', function($q){
                        $q->where('is_active', 1);
                    })->where('user_main_id', auth()->user()->main_id)
                    ->where('is_active', 1)
                    ->get();
        
        $items = Item::with('main_item', 'categories')
                ->whereHas('categories', function($q){
                    $q->where('is_active', 1);
                })
                ->whereNull('item_id')
                ->where('user_main_id', auth()->user()->main_id)
                ->get()
                ->map(function($item){
                    $item['addon_count'] = 0;
                    if(!empty($item->item_id)){
                        $item['addon_count'] = ItemAddon::where('item_id', $item->item_id)->count();
                    }else{
                        $item['addon_count'] = ItemAddon::where('item_id', $item->id)->count();
                    }
                    return $item;
                });
        
        $taxes = Tax::where('user_main_id', auth()->user()->main_id)
                ->where('is_active', 1)
                ->get()
                ->keyBy('id');

        $customers = Customer::select('id', 'name')->where('user_main_id', auth()->user()->main_id)->get();
        $payment_methods = PaymentMethod::find(is_array($canteen->pos_payment_methods_ids) ? $canteen->pos_payment_methods_ids : []);
        $payment_statuses = PaymentStatus::get();
        $order_statuses = OrderStatus::where('is_active', 1)->get();

        return view('canteen.pages.pos.index', compact('canteen', 'categories', 'items', 'customers', 'taxes', 'payment_methods', 'payment_statuses', 'order_statuses'));
    }

    public function orderStore(Request $request){

        $request->validate([
            'customer_id' => ['nullable', 'numeric'],
            'order_status_id' => ['required', 'numeric'],
            'payment_method_id' => ['required', 'numeric'],
            'payment_status_id' => ['required', 'numeric'],
            'restaurant_instruction' => ['nullable', 'string'],
            'cartItem' => ['required', 'array'],
            'cartItem.*.item_id' => ['required', 'numeric'],
            'cartItem.*.quantity' => ['required', 'numeric'],
            'cartItem.*.discount' => ['required', 'numeric'],
        ]);

        $reqOrderObj = new stdClass();
        $reqOrderObj->main_id = auth()->user()->main_id;
        $reqOrderObj->currency = auth()->user()->currency;
        $reqOrderObj->customer_id = $request->customer_id;
        $reqOrderObj->ordered_at = Carbon::now()->toDateTimeString();
        $reqOrderObj->restaurant_instruction = $request->restaurant_instruction;
        $reqOrderObj->need_cutlery = $request->need_cutlery;

        if (auth()->user()->can('orders.update-status')) {
            $reqOrderObj->order_status_id = $request->order_status_id;
        }else{
            $reqOrderObj->order_status_id = 1;
        }

        $reqOrderObj->payment_method_id = $request->payment_method_id;

        if (auth()->user()->can('others.accept-payment')) {
            $reqOrderObj->payment_status_id = $request->payment_status_id;
        }else{
            $reqOrderObj->payment_status_id = 1;
        }

        $reqOrderObj->order_placed_by = auth()->user()->id;
        $reqOrderObj->cartItem = $request->cartItem;

        $response = $this->ou_store_order($reqOrderObj);

        if(isset($response["order_id"]) && !empty($response["order_id"])){

            $order = Order::find($response["order_id"]);

            return response()->json([
                'success' => true,
                'data' => $order,
                'html_response' => view('canteen.pages.pos.order-success', compact('order'))->render(),
            ]);

        }

        return response()->json([
            'success' => false,
            'message' => 'Oops! Something didn’t go as planned. Please try again in a moment.',
        ]);

    }

}
