<?php

namespace App\Http\Controllers\CANTEEN;

use App\Http\Controllers\Controller;
use App\Models\ItemBadge;
use App\Traits\FileUpload;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Validation\Rule;

class ItemBadgeController extends Controller implements HasMiddleware
{
    use FileUpload;

    public static function middleware(): array
    {
        return [
            new Middleware('permission:item-badges.read', only: ['index']),
            new Middleware('permission:item-badges.create', only: ['create', 'store']),
            new Middleware('permission:item-badges.update', only: ['edit','update']),
            new Middleware('permission:item-badges.delete', only: ['destroy']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        //
        $rowsQuery = ItemBadge::sortable()
            ->where('user_main_id', auth()->user()->main_id);
        
        if($request->query('query')){

            $searchQuery = $request->query('query');

            $rowsQuery->where(function($query) use($searchQuery){
                $query->where('uid', 'like', '%' . $searchQuery . '%')
                ->orWhere('name', 'like', '%' . $searchQuery . '%')
                ->orWhere('position', 'like', '%' . $searchQuery . '%');
            });
        }

        $rows = $rowsQuery->paginate(setting("data_per_page"));

        return view('canteen.pages.itemBadge.index', compact('rows'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        return view('canteen.pages.itemBadge.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
        $request->validate([
            'name' => ['required', 'max:100', Rule::unique('item_badges')->where(function ($query) {
                return $query->where('user_main_id', auth()->user()->main_id)
                ->whereNull('deleted_at');
            })],
            'image' => ['required', 'image', 'max:512'],
        ]);

        $item_badge = new ItemBadge();
        $item_badge->user_main_id = auth()->user()->main_id;

        $item_badge->name = $request->name;

        if($request->has('image')){
            $imageURL = $this->uploadFile($request->image, 'itembadges');
            if($imageURL['success'] == true){
                $item_badge->image = $imageURL['data'];
            }
        }
        
        $item_badge->is_active = !empty($request->is_active) ? 1 : 0;
        $item_badge->position = $request->position;
        $item_badge->save();
        
        \Session::flash("message", "Item Badge created successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/item-badges'),
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $subdomain, string $id)
    {
        //
        $item_badge = ItemBadge::where('user_main_id', auth()->user()->main_id)->findOrFail($id);

        return view('canteen.pages.itemBadge.edit', compact('item_badge'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $subdomain, string $id)
    {
        //
        $request->validate([
            'name' => ['required', 'max:100', Rule::unique('item_badges')->where(function ($query) {
                return $query->where('user_main_id', auth()->user()->main_id)
                ->whereNull('deleted_at');
            })->ignore($id)],
            'image' => ['nullable', 'image', 'max:512'],
        ]);

        $item_badge = ItemBadge::where('user_main_id', auth()->user()->main_id)->find($id);

        $item_badge->name = $request->name;

        if($request->has('image')){
            $imageURL = $this->uploadFile($request->image, 'itembadges');
            if($imageURL['success'] == true){
                $this->deleteFile($item_badge->image);
                $item_badge->image = $imageURL['data'];
            }
        }
        
        $item_badge->is_active = !empty($request->is_active) ? 1 : 0;
        $item_badge->position = $request->position;
        $item_badge->save();
        
        \Session::flash("message", "Item Badge updated successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/item-badges'),
        ]);
    }

    public function beforeDelete(string $subdomain, string $id){

        return response()->json([
            'deletion_allowed' => true,
            'message' => ''
        ]);

        // return response()->json([
        //     'deletion_allowed' => false,
        //     'message' => 'This tax is associated with an item. You cannot remove it.',
        // ]);

    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $subdomain, string $id)
    {
        //
        $item_badge = ItemBadge::where('user_main_id', auth()->user()->main_id)->findOrFail($id);
        $item_badge->delete();

        \Session::flash("message", "Item Badge deleted successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/item-badges'),
        ]);
    }
}
