<?php

namespace App\Http\Controllers\CANTEEN;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Traits\AppMails;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class EmployeeController extends Controller implements HasMiddleware
{
    use AppMails;

    public static function middleware(): array
    {
        return [
            new Middleware('permission:employees.read', only: ['index']),
            new Middleware('permission:employees.create', only: ['create', 'store']),
            new Middleware('permission:employees.update', only: ['edit','update']),
            new Middleware('permission:employees.delete', only: ['destroy']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        //
        $rowsQuery = User::sortable()->where('type', 'employee')
            ->where('main_id', auth()->user()->main_id);
        
        if($request->query('query')){

            $searchQuery = $request->query('query');

            $rowsQuery->where(function($query) use($searchQuery){
                $query->where('uid', 'like', '%' . $searchQuery . '%')
                ->orWhere('name', 'like', '%' . $searchQuery . '%')
                ->orWhere('email', 'like', '%' . $searchQuery . '%')
                ->orWhere('mobile_number', 'like', '%' . $searchQuery . '%')
                ->orWhere('mobile_number', 'like', '%' . $searchQuery . '%')
                ->orWhere('address', 'like', '%' . $searchQuery . '%');
            });
        }

        $rows = $rowsQuery->paginate(setting("data_per_page"));

        return view('canteen.pages.employee.index', compact('rows'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        return view('canteen.pages.employee.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
        $request->merge(['mobile_number' => str_replace([' ', '-'],'', $request->mobile_number)]);

        //
        $request->validate([
            'designation' => ['required', 'max:100'],
            'name' => ['required', 'max:100'],
            'email' => ['required', 'email:rfc,dns', 'max:100', Rule::unique('users')->where(function ($query){
                return $query->where('main_id', auth()->user()->main_id)
                    ->whereNull('deleted_at');          
            })],
            'mobile_number' => ['required', Rule::unique('users')->where(function ($query){
                return $query->where('main_id', auth()->user()->main_id)
                    ->whereNull('deleted_at');          
            })],
            'address' => ['required', 'max:120'],
        ]);

        $password = "Pass@". rand(1000, 9999);

        $employee = new User();
        $employee->main_id = auth()->user()->main_id;
        
        $employee->user_name = uniqid();
        $employee->type = "employee";

        $employee->designation = $request->designation;
        $employee->name = $request->name;
        $employee->email = $request->email;
        $employee->mobile_number = $request->mobile_number;
        $employee->address = $request->address;

        $employee->otp_verified_at = Carbon::now()->toDateString();
        $employee->password = Hash::make($password);
        $employee->is_verified = 1;
        $employee->country = "BD";

        $employee->save();
        
        $business = User::find($employee->main_id);

        $this->am_employee_welcome_mail($business, $employee, $password);

        \Session::flash("message", "Employee created successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/employees'),
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $subdomain, string $id)
    {
        //
        $employee = User::where('main_id', auth()->user()->main_id)->findOrFail($id);

        return view('canteen.pages.employee.edit', compact('employee'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $subdomain, string $id)
    {
        //
        $request->merge(['mobile_number' => str_replace([' ', '-'],'', $request->mobile_number)]);

        //
        $request->validate([
            'designation' => ['required', 'max:100'],
            'name' => ['required', 'max:100'],
            'email' => ['required', 'email:rfc,dns', 'max:100', Rule::unique('users')->where(function ($query){
                return $query->where('main_id', auth()->user()->main_id)
                    ->whereNull('deleted_at');          
            })->ignore($id)],
            'mobile_number' => ['required', Rule::unique('users')->where(function ($query){
                return $query->where('main_id', auth()->user()->main_id)
                    ->whereNull('deleted_at');          
            })->ignore($id)],
            'address' => ['required', 'max:120'],
        ]);

        $password = "Pass@". rand(1000, 9999);

        $employee = User::where('main_id', auth()->user()->main_id)->findOrFail($id);
        
        // $employee->user_name = uniqid();
        // $employee->type = "employee";

        $employee->designation = $request->designation;
        $employee->name = $request->name;
        $employee->email = $request->email;
        $employee->mobile_number = $request->mobile_number;
        $employee->address = $request->address;

        if($request->filled('password')){
            $employee->password = Hash::make($password);
        }

        $employee->save();
        
        \Session::flash("message", "Employee updated successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/employees'),
        ]);
    }

    public function beforeDelete(string $subdomain, string $id){

        return response()->json([
            'deletion_allowed' => true,
            'message' => ''
        ]);

        // return response()->json([
        //     'deletion_allowed' => false,
        //     'message' => 'This tax is associated with an item. You cannot remove it.',
        // ]);

    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $subdomain, string $id)
    {
        //
        //
        $employee = User::where('main_id', auth()->user()->main_id)->findOrFail($id);
        $employee->delete();

        \Session::flash("message", "Employee deleted successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/employees'),
        ]);

    }
}
