<?php

namespace App\Http\Controllers\CANTEEN;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class CustomerController extends Controller implements HasMiddleware
{

    public static function middleware(): array
    {
        return [
            new Middleware('permission:customers.read', only: ['index']),
            new Middleware('permission:customers.create', only: ['create', 'store']),
            new Middleware('permission:customers.update', only: ['edit','update']),
            new Middleware('permission:customers.delete', only: ['destroy']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        //
        $rowsQuery = Customer::sortable()
            ->where('type', 'customer')
            ->where('user_main_id', auth()->user()->main_id);

        if($request->query('query')){

            $searchQuery = $request->query('query');

            $rowsQuery->where(function($query) use($searchQuery){
                $query->where('uid', 'like', '%' . $searchQuery . '%')
                ->orWhere('name', 'like', '%' . $searchQuery . '%')
                ->orWhere('name', 'like', '%' . $searchQuery . '%')
                ->orWhere('email', 'like', '%' . $searchQuery . '%')
                ->orWhere('mobile_number', 'like', '%' . $searchQuery . '%')
                ->orWhere('address', 'like', '%' . $searchQuery . '%')
                ->orWhere('country', 'like', '%' . $searchQuery . '%');
            });

        }
        
        $rows = $rowsQuery->paginate(setting("data_per_page"));

        return view('canteen.pages.customers.index', compact('rows'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        return view('canteen.pages.customers.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->merge(['mobile_number' => str_replace([' ', '-'],'', $request->mobile_number)]);

        //
        $request->validate([
            'name' => ['required', 'max:100'],
            'email' => ['nullable', 'email:rfc,dns', 'max:100'],
            'mobile_number' => ['required', Rule::unique('customers')->where(function ($query){
                return $query->where('user_main_id', auth()->user()->main_id)
                    ->whereNull('deleted_at');          
            })],
            'address' => ['nullable', 'max:120'],
        ]);

        $password = "Pass@". rand(1000, 9999);

        $customer = new Customer();
        $customer->user_main_id = auth()->user()->main_id;

        $customer->name = $request->name;
        $customer->email = $request->email;
        $customer->mobile_number = $request->mobile_number;
        $customer->address = $request->address;

        $customer->otp_verified_at = Carbon::now()->toDateString();
        $customer->password = Hash::make($password);
        $customer->is_verified = 1;
        $customer->country = "BD";

        $customer->save();
        
        \Session::flash("message", "Customer created successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/customers'),
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $subdomain, string $id)
    {
        //
        $customer = Customer::where('user_main_id', auth()->user()->main_id)->where('type', 'customer')->findOrFail($id);

        return view('canteen.pages.customers.edit', compact('customer'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $subdomain, string $id)
    {

        $request->merge(['mobile_number' => str_replace([' ', '-'],'', $request->mobile_number)]);

        //
        $request->validate([
            'name' => ['required', 'max:100'],
            'mobile_number' => ['required', Rule::unique('customers')->where(function ($query){
                return $query->where('user_main_id', auth()->user()->main_id)
                    ->whereNull('deleted_at');
            })->ignore($id)],
            'email' => ['nullable', 'email:rfc,dns', 'max:100'],
            'address' => ['nullable', 'max:120'],
        ]);

        $customer = Customer::where('user_main_id', auth()->user()->main_id)->where('type', 'customer')->find($id);
        $customer->name = $request->name;
        $customer->email = $request->email;
        $customer->mobile_number = $request->mobile_number;
        $customer->address = $request->address;

        $customer->save();
        
        \Session::flash("message", "Customer updated successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/customers'),
        ]);
    }

    public function beforeDelete(string $subdomain, string $id){

        return response()->json([
            'deletion_allowed' => true,
            'message' => ''
        ]);

        // return response()->json([
        //     'deletion_allowed' => false,
        //     'message' => 'This tax is associated with an item. You cannot remove it.',
        // ]);

    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $subdomain, string $id)
    {
        //
        $customer = Customer::where('user_main_id', auth()->user()->main_id)->where('type', 'customer')->findOrFail($id);
        $customer->delete();

        \Session::flash("message", "Customer deleted successfully");

        return response()->json([
            'success' => true,
            'redirect_url' => url('/app/customers'),
        ]);
    }
}
